#   G-Code generator for simple PCB etching
#   Author Fred LaPlante, 8Oct2015
#
#   This program was originally intended to create PCBs for use in 'Manhattan' style construction.
#   Thus the board to be etched is assumed to be laid out as a grid of cells
#   identified by rows and columns numbered from 1 to N along both the X and Y axis.
#
#   Isolated areas will have a 'moat' ecthed along their edges.
#   Vias will be marked with a small pit at the center of the specified cell.
#
#   The user may also specify that the boarder around the board is to be cut all the way
#   through except for small 'tabs' at the corners which may later be cut by hand.
#   
#   mods:
#   07Jul2017 - Polygons and Rectangles now use transparent fill, no need to use lines to draw
#   04Jul2017 - ver 5, refactored into separate gui, drawing, gcode modules
#   03Jul2017 - added dialog to set gcode folder, all 'G' codes are now CAPS
#   02Jul2017 - Added hot keys for file Open, Save, & Quit.
#   01Jul2017 - ver 4, added straight lines, added 'object select', added 'object delete'
#               removed 'zig-zag', added polygons, moved stuff to menus, seperate gcode from draw
#               file open & save, integrate via, refine drawing toolbar, added border, added
#               'clear gcode', added edit menu with 'selectAll', gcode generation now generated
#               from canvas object list
#   19Apr2017 - ver 3, Add support for 'zig-zag' strips - incomplete
#   31Jan2016 - Add file Open dialog to menu
#   20Jan2016 - Get confirm to exit without 'finish' and 'Save'.
#   02Jan2016 - Allow canvas to expand as needed for both rows and columns
#               Remove 'F' codes from all G00 movements
#               Widen PCB parameter fields
#   20Nov2015 - ver 2, add GUI inteface, persistent changeable parameters and board layout preview
#   16Nov2015 - vias now marked with depth equal to PCB thickness for visability.
#   13Nov2015 - spelling, improved labels for isolated areas
#   30Oct2015 - cells may have x & y length be simple multiples of the base grid size.
#   19Oct2015 - vias can now be entered, add means to cut board free (leave corner tabs)
#   18Oct2015 - allow finer placement of pads, add border

from tkinter import *
from tkinter.ttk import *
from tkinter import messagebox
import os

import draw
import gcode
import flosLib
from config import *

params = fetch_params()

class SmplPCB(flosLib.Framework):
    def __init__(self, root):
        super().__init__(root)
        self.root = root
        self.create_gui()

    def create_gui(self):
        self.create_gcode_area()
        self.create_drawing_area()
        self.create_menu()  # has to come last as menus refer to parts of above widgets

    def create_menu(self):
        #print ('making menus')
        self.menuBar = Menu(self.root)
        menu_items = (
            'File ~ &New/Ctrl+n/self.new_file, &Open/Ctrl+o/self.open_file, &Save/Crrl+s/self.save_file, SaveAs//self.saveas_file, Exit/Ctrl+q/self.exit_app',
            'Edit ~ SelectAll//self.selectAll',
            'Options ~ Program//self.programOpts, Machine//self.machineOpts, GCode//self.gcodeOpts, PCB//self.pcbOpts',
            'G-Code ~ Clear//self.delGcode, sep, Generate//self.genGcode',
            'Help ~ Help//self.help, sep, Show Obj//self.show, sep, About//self.about'
            )
        self.build_menu(menu_items)  # here's our framework, flosLib, in action

    def create_gcode_area(self):
        row = 3
        col = 30
        self.gcode = gcode.Gcode(self.root, row, col)
        self.gcode.create_editor()
        self.make_kybd_shortcuts()
        self.gcode.make_popupMenu()
        
    def create_drawing_area(self):
        row = 1
        col = 0
        self.draw = draw.Draw(self.root, row, col)
        self.draw.keepRef(self.gcode)
        self.draw.make_tool_bar()
        self.draw.make_canvas()

    def make_kybd_shortcuts(self):
        self.gcode.content.bind('<Control-n>', self.new_file)
        self.gcode.content.bind('<Control-o>', self.open_file)
        self.gcode.content.bind('<Control-s>', self.save_file)
        
    # ---- Menu Event Handlers ---- #
    #- File Menu -#
    def new_file(self, event=None):
        self.draw.doUnselect()
        self.gcode.filenew()
    
    def open_file(self, event=None):
        self.draw.doUnselect()
        self.gcode.fileopen()
        
    def save_file(self, event=None):
        print('saving')
        self.draw.doUnselect()
        self.gcode.filesave()

    def saveas_file(self):
        print('saving as ...')
        self.draw.doUnselect()
        self.gcode.filesaveas()
                
    def exit_app(self, event=None):
        #print('Quitting')
        self.draw.doUnselect()
        if (self.gcode.saved == False):
            if messagebox.askokcancel("Quit", "Do you want to 'Save' your work before quitting?"):
                self.on_save()
        root.destroy()

    #- Edit Menu -#
    def selectAll(self):
        self.draw.doSelectAll()

    #- Options Menu -#
    def programOpts(self):
        self.draw.doUnselect()
        do_programOpts(self.root)
        
    def machineOpts(self):
        self.draw.doUnselect()
        do_machineOpts(self.root)
            
    def gcodeOpts(self):
        self.draw.doUnselect()
        do_gcodeOpts(self.root)

    def pcbOpts(self):
        self.draw.doUnselect()
        do_pcbOpts(self.root)
        self.draw.resize_canvas()

    #- Gcode Menu -#
    def delGcode(self):
        #print('clearing G-code')
        self.draw.doUnselect()
        #self.gcode.delAllGcode()
        self.gcode.filenew()
        
    def genGcode(self):
        #print('generating G-code')
        self.draw.doUnselect()
        self.delGcode()
        self.draw.make_gcode()
        self.gcode.saved = False
        
    #- Help Menu -#
    def help(self):
        self.draw.doUnselect()
        messagebox.showinfo("Help", "Not yet Available")

    def show(self):
        self.draw.doUnselect()
        objList = self.draw.get_object_list()
        if len(objList) > 0:
            msg = objList
        else:
            msg = 'No objects currently exist.'
        messagebox.showinfo("Drawn Object List", msg)

    def about(self):
        self.draw.doUnselect()
        messagebox.showinfo("About smplPCB","smplPCB\nVersion 5.0\n\nFred LaPlante")

    #- No Menu -#
    def notDone(self):
        self.draw.doUnselect()
        messagebox.showerror('Not implemented', 'Not yet available')

if __name__ == '__main__':
    root = Tk()
    PROGRAM_NAME = " smplPCB "
    root.title(PROGRAM_NAME)
    params = fetch_params()

    app = SmplPCB(root)
    root.mainloop()

