from tkinter import *
from tkinter.ttk import *

from config import *
from gcode import *

class Draw():
    def __init__(self, root, row, col):
        #print('prepping drawing area')
        
        self.root = root
        self.base_row = row
        self.base_col = col
        
        # prepare stores for drawn items.
        self.chrome = list()
        self.items = list()
        self.tempIds = list()
        self.selected = list()
        self.rectPts = list()
        self.polygonPts = list()
        self.borderPts = list()

    def keepRef(self, gcode):
        self.gcode = gcode
        
    def make_tool_bar(self):
        print('creating tool bar')
        modeFrame = Frame(self.root, borderwidth=2, relief='ridge')
        modeFrame.grid(row=self.base_row, column=self.base_col, sticky='W')

        self.mode = 'select'
        self.inptType = StringVar()
        self.inptType.set(self.mode)
        Radiobutton(modeFrame, text='Select', variable=self.inptType, value='select', command=self.on_inptChange, compound='top').grid(row=2, column=1, sticky='W')
        Radiobutton(modeFrame, text='Line', variable=self.inptType, value='line', command=self.on_inptChange, compound='top').grid(row=2, column=3, sticky='W')
        Radiobutton(modeFrame, text='Rectangle', variable=self.inptType, value='rect', command=self.on_inptChange, compound='top').grid(row=2, column=5, sticky='W')
        Radiobutton(modeFrame, text='Polygon', variable=self.inptType, value='poly', command=self.on_inptChange, compound='top').grid(row=2, column=7, sticky='W')
        Radiobutton(modeFrame, text='Via', variable=self.inptType, value='via', command=self.on_inptChange, compound='top').grid(row=2,column=9, sticky='W')
        Radiobutton(modeFrame, text='Border', variable=self.inptType, value='bord', command=self.on_inptChange, compound='top').grid(row=2,column=11, sticky='W')

    def make_canvas(self):
        #print('creating drawing area')
        params = get_params()
        #print('crnt params=',params)
        canvasFrame = Frame(self.root, borderwidth=2, relief='ridge')
        canvasFrame.grid(row=self.base_row+2, column=self.base_col, sticky=N+W)
        Label(canvasFrame, text='Board Layout Preview').grid(row=0, column=0, sticky=N)

        self.canvas = Canvas(canvasFrame, bg=params['bg_color'])
        self.canvas.grid(row=2, column=0, sticky=N+W)
        self.resize_canvas()

        self.canvas.bind("<Button-1>", self.on_mouseDown)
        self.canvas.bind("<Button1-Motion>", self.on_mouseDownMotion)
        self.canvas.bind("<Button1-ButtonRelease>", self.on_mouseUp)
        #### need to find a better bind level ####
        self.root.bind("<KeyPress-Escape>", self.on_escape)
        self.root.bind("<KeyPress-Delete>", self.on_delete)
        #root.bind("<KeyPress-Return>", self.on_return)
        #root.bind("<Any-KeyPress>", self.on_anyKey)

    def resize_canvas(self):
        #print('resizing canvas')
        params = get_params()
        canvWid = int(((params['nCols'] * params['cellSize']) + (2 * params['xMargin']))*params['scale'])
        canvHt = int(((params['nRows'] * params['cellSize']) + (2* params['yMargin']))*params['scale'])        
        self.canvas.configure(width=canvWid, height=canvHt)
        # clean out old lines and numbers
        for objId in self.chrome:
            self.canvas.delete(objId)
        # add new stuff
        self.drawGrid()
        self.drawColNmbrs()
        self.drawRowNmbrs()
        
    def drawColNmbrs(self):
        #print('draw col nmbrs')
        params = get_params()
        y = (((params['nRows']*params['cellSize'])+(2*params['yMargin'])) * params['scale']) - (params['cellSize']/2.0)*params['scale']
        for n in range(1, params['nCols']+1, 1):
            x = int((((n-1)*params['cellSize'])+params['xMargin']) * params['scale']) + int((params['cellSize']/2)*params['scale'])
            colNumId = self.canvas.create_text(x,y, text=str(n))
            self.canvas.addtag_withtag('cols', colNumId)
            self.chrome.append(colNumId)

    def drawRowNmbrs(self):
        #print('draw row nmbrs')
        params = get_params()
        x = (params['cellSize']/2.0)*params['scale']
        for n in range(1, params['nRows']+1, 1):
            yAdj = (((params['nRows']*params['cellSize'])+(2*params['yMargin'])) * params['scale'])
            y = yAdj - int((((n-1)*params['cellSize'])+params['xMargin']) * params['scale']) - int((params['cellSize']/2)*params['scale'])
            rowNumId = self.canvas.create_text(x,y, text=str(n))
            self.canvas.addtag_withtag('rows', rowNumId)
            self.chrome.append(rowNumId)
            
    def drawGrid(self):
        params = get_params()
        startY = int(params['yMargin']*params['scale'])
        endY = startY + int((params['nRows']*params['cellSize']) * params['scale'])
        for x in range(0, params['nCols']+1, 1):
            startX = endX = int(((x*params['cellSize'])+params['xMargin']) * params['scale'])
            #print ('startX=',startX, '  startY=',startY, '  endX=',endX, 'endY=',endY)
            vLineId = self.canvas.create_line(startX, startY, endX, endY, fill=params['grid_color'], dash=(1,3))
            self.canvas.addtag_withtag('grid', vLineId)
            self.chrome.append(vLineId)
        #print('-')
        startX = int(params['xMargin']*params['scale'])
        endX = startX + int((params['nCols']*params['cellSize']) * params['scale'])
        for y in range(0, params['nRows']+1, 1):
            startY = endY = int(((y*params['cellSize'])+params['yMargin']) * params['scale'])
            #print ('startX=',startX, '  startY=',startY, '  endX=',endX, 'endY=',endY)
            hLineId = self.canvas.create_line(startX, startY, endX, endY, fill=params['grid_color'], dash=(1,3))
            self.canvas.addtag_withtag('grid', hLineId)
            self.chrome.append(hLineId)

    def canv2cellX(self, canvX):
        params = get_params()
        #print('canvX= ',canvX)
        xDim = canvX - (params['xMargin'] * params['scale'])
        #print('xDim= ', xDim)
        cellNmbr = int(xDim // (params['cellSize'] * params['scale'])) + 1
        #print('cell X # = ', cellNmbr)
        return cellNmbr

    def canv2cellY(self, canvY):
        params = get_params()
        #print('canvY= ',canvY)
        yDim = canvY - (params['yMargin'] * params['scale'])
        #print('yDim= ', yDim)
        cellNmbr = int(params['nRows']) - int(yDim // (params['cellSize'] * params['scale']))
        #print('cell Y # = ', cellNmbr)
        return cellNmbr
    
    #---- Canvas Drawing Routines ----#       
    def on_inptChange(self):
        self.doUnselect()
        self.mode = self.inptType.get()

    def on_objSelect(self, event):
        self.doUnselect()
        # then identify the new selection
        self.objX = self.canvas.canvasx(event.x)
        self.objY = self.canvas.canvasy(event.y)
        objId = self.canvas.find_closest(self.objX,self.objY, halo=1)
        self.selected.append(objId)
        self.mark_selected(objId)

    def doSelectAll(self):
        self.selected = self.canvas.find_withtag('smpl')
        #print("'selected' list: ",self.selected)
        selColor = getParam('select_color')
        for objId in self.selected:
            self.mark_selected(objId)
                     
    def mark_selected(self, objId):
        selColor = getParam('select_color')
        tags = self.canvas.gettags(objId)
        print(tags)
        if (tags.count('line') > 0):
            self.canvas.itemconfigure(objId, fill=selColor)
        elif (tags.count('rect') > 0) or (tags.count('poly') > 0) or (tags.count('bord') > 0)  or (tags.count('via') > 0):
            self.canvas.itemconfigure(objId, outline=selColor)
        else:
            pass

    def doUnselect(self):
        params = get_params()
        # reset any previously selected objects
        for objId in self.selected:
            tags = self.canvas.gettags(objId)
            if (tags.count('line') > 0):
                self.canvas.itemconfigure(objId, fill=params['line_color'])
            elif (tags.count('via') > 0):
                self.canvas.itemconfigure(objId, outline=params['via_color'])
            elif (tags.count('rect') > 0) or (tags.count('poly') > 0):
                self.canvas.itemconfigure(objId, outline=params['line_color'])
            elif (tags.count('bord') > 0):
                self.canvas.itemconfigure(objId, outline=params['border_color'])
            else:
                pass
        self.selected = []
        
    #def on_anyKey(self, event):
    #    print('pressed ', event.keysym, '_num=',event.keysym_num)
        
    def on_delete(self, event):
        #print('delete key pressed')
        #print(self.selected)
        for objId in self.selected:
            #print(objId)
            self.canvas.delete(objId)
        self.selected = list()
        
    def on_escape(self, event):
        print('escape key pressed')
        self.doUnselect()
        if (self.x1 == self.x0) and (self.y1 == self.y0):
            return
        if (self.mode == 'poly'):
            print('poly drawing completed')
            point = (self.x1,self.y1)
            self.polygonPts.append(point)
            self.cleanTempIds()
            self.drawPolygon()

    def on_mouseDown(self, event):
        if (self.mode != 'select'):
            self.doUnselect()
        
        gridSize = getParam('gridSize')
        if (self.mode == 'via'):
            self.x0 = self.canvas.canvasx(event.x)
            self.y0 = self.canvas.canvasy(event.y)
        else:
            self.x0 = self.canvas.canvasx(event.x, gridSize)
            self.y0 = self.canvas.canvasy(event.y, gridSize)        
        print('mouse-down: x0=',self.x0,', y0=',self.y0)
        
        if (self.mode == 'rect'):
            print('rect line start')
            point = (self.x0,self.y0)
            self.rectPts.append(point)
        if (self.mode == 'poly'):
            print('poly line start')
            point = (self.x0,self.y0)
            self.polygonPts.append(point)
        elif (self.mode == 'bord'):
            point = (self.x0,self.y0)
            self.borderPts.append(point)

    def cleanTempIds(self):
        # remove any past temp line
        try:
            for wid in self.tempIds:
                self.canvas.delete(wid)
        except (AttributeError, NameError):
            pass
            
    def on_mouseDownMotion(self, event):
        if ((self.mode == 'line') or (self.mode == 'bord') or (self.mode == 'poly')):
            self.cleanTempIds()
            
        # draw temp line(s) whenever mouse is in motion
        gridSize = getParam('gridSize')
        crntX = self.canvas.canvasx(event.x, gridSize)
        crntY = self.canvas.canvasy(event.y, gridSize)
        #drawColor = getParam('draw_color')
        drawColor = 'red'
        #print('mode: ',self.mode)
        if (self.mode == 'via'):
            pass
        elif ((self.mode == 'line')):
            self.tempId = self.canvas.create_line(self.x0,self.y0, crntX,crntY, fill=drawColor, width=2)
        elif (self.mode == 'rect') or (self.mode == 'bord'):
            self.tempId = self.canvas.create_rectangle(self.x0,self.y0, crntX,crntY, outline=drawColor, width=2)
        elif (self.mode == 'poly'):
            tempPts = self.polygonPts + [crntX,crntY]
            self.tempId = self.canvas.create_polygon(tempPts, outline=drawColor, fill='', width=2)
        self.canvas.addtag_withtag('temp', self.tempId)
        self.tempIds.append(self.tempId)
        
    def on_mouseUp(self, event):
        params = get_params()
        if (self.mode == 'select'):
            #print('ready to select an object');
            self.on_objSelect(event)
        else:
            self.doUnselect()
            if (self.mode == 'via'):
                self.x1 = self.canvas.canvasx(event.x)
                self.y1 = self.canvas.canvasy(event.y)
                cellX = self.canv2cellX(self.x1)
                if (cellX < 1) or (cellX > params['nCols']):
                    return
                cellY = self.canv2cellY(self.y1)
                if (cellY < 1) or (cellY > params['nRows']):
                    return
                self.lastX = cellX
                self.lastY = cellY
                #print('mouse btn up in cell: ', cellX, cellY)
                self.drawVia(cellX,cellY)
            else:
                self.x1 = self.canvas.canvasx(event.x, params['gridSize'])
                self.y1 = self.canvas.canvasy(event.y, params['gridSize'])
                if (self.mode == 'line'):
                    self.cleanTempIds()
                    self.drawLine()
                elif (self.mode == 'poly'):
                    if ((self.x1 == self.x0) and (self.y1 == self.y0)):
                        print('saving end pt of poly')
                        point = (self.x1,self.y1)
                        self.polygonPts.append(point)
                elif (self.mode == 'rect'):
                    self.cleanTempIds()
                    point = (self.x1,self.y1)
                    self.rectPts.append(point)
                    self.drawRectangle()
                elif (self.mode == 'bord'):
                    self.cleanTempIds()
                    point = (self.x1,self.y1)
                    self.borderPts.append(point)
                    self.drawBorder()
            
    def drawLine(self):
        #print('grid spacing=',self.gridSize)
        #print('x0=',self.x0,' y0=',self.y0, '; x1=',self.x1,' y1=',self.y1)
        lineColor = getParam('line_color')
        wid = self.canvas.create_line(self.x0,self.y0, self.x1,self.y1, fill=lineColor, width=2, tags=('smpl','line'))

    def drawRectangle(self):
        lineColor = getParam('line_color')
        #print(self.borderPts)
        tl = self.rectPts.pop(0)
        br = self.rectPts.pop(0)
        self.canvas.create_rectangle(br[0],br[1], tl[0],tl[1], outline=lineColor,fill='', width=2, tags=('smpl','rect'))
        self.rectPts = list()

    def drawPolygon(self):
        #print(self.polygonPts)
        lineColor = getParam('line_color')
        wid = self.canvas.create_polygon(self.polygonPts, outline=lineColor, fill='', width=2, tags=('smpl','poly'))
        self.polygonPts = list()

    def drawVia(self, cellX,cellY):
        params = get_params()
        xCentr = ((int(cellX) - 1) * float(params['cellSize'])) + float(params['xMargin']) + float(params['cellSize'])/2.0
        yCentr = ((int(cellY) - 1) * float(params['cellSize'])) + float(params['yMargin']) + float(params['cellSize'])/2.0
        yCentr = (((params['nRows']*params['cellSize'])+(2*params['yMargin'])) - yCentr)
        radius = (params['cellSize']*0.1*params['scale'])
        x0 = float((xCentr * params['scale']) + radius)                                                                   
        y0 = float((yCentr * params['scale']) + radius)                                                                   
        xn = float((xCentr * params['scale']) - radius)                                                                   
        yn = float((yCentr * params['scale']) - radius)                                                                   
        wid = self.canvas.create_oval(x0,y0, xn,yn, outline=params['via_color'], fill='', width=1, tags=('smpl','via'))

    def drawBorder(self):
        brdrColor = getParam('border_color')
        #print(self.borderPts)
        tl = self.borderPts.pop(0)
        br = self.borderPts.pop(0)
        self.canvas.create_rectangle(br[0],br[1], tl[0],tl[1], outline=brdrColor,fill='', width=2, tags=('smpl','bord'))
        self.borderPts = list()

    def get_object_list(self):
        objList = self.canvas.find_withtag('smpl')
        return objList

    #----- gcode generation -----#
    def make_gcode(self):
        self.gcode.initGcode()
        self.gcode.makeLineGcode(self.canvas)
        self.gcode.makeRectangleGcode(self.canvas)       
        self.gcode.makePolygonGcode(self.canvas)       
        self.gcode.makeViaGcode(self.canvas)        
        self.gcode.makeBorderGcode(self.canvas)       
        self.gcode.goHome()
        self.finished = True;
    
if __name__ == '__main__':

    root = Tk()
    fetch_params()
    app = Draw(root, 0,0)
    root.mainloop()
