function inherit(p) {
	// from "JavaScript: the definitive guide", 6th ed, p.119
	if (p == null) throw TypeError();						// p must be non-null object
	if (Object.create) return Object.create(p);	// use it if you got it!
	// else																			
	var t = typeof p;														// otherwise do more type checking
	if (t!== "object" && t!== "function") throw TypeError();	
	function f() {};														// define a dummy constructor
	f.prototype = p;														// set its prototype to p
	return new f();															// use f() to create an 'heir' of p
}

//-----------------------------
$.fn.toggleNextUL = function() { 
  return this.each( function() {
    $(this).next('ul').toggle();
  });
}

//-----------------------------
function makeNav () {
	// based on "Stupid Simple jQuery Accordian" by Ryan Stemkoski, 2009
	$(".navHeading").bind('click',null,function() {
		$(".navSelected").removeClass("navSelected").addClass("navUnselected")
		$(this).removeClass("navUnselected").addClass("navSelected");
		$(".navContent").slideUp();	
		$(this).next(".navContent").slideDown();
	});
	
	$(".navContent").hide();
	$(".navHeading").addClass("navUnselected");
	$("nav #defaultOpen").trigger("click");
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
function Chart (id, name, width, height, bgColor) {
	this.name = name; this.id = id;
	var canvas = document.getElementById(this.id);
	this.ctx = canvas.getContext('2d');
	this.ctx.clearRect(0,0,width,height);
	this.ctx.fillStyle = bgColor;
	this.ctx.fillRect(0,0,width,height);
};

Chart.prototype.makeBox = function(top,left, wid,ht, maxVal,minVal,inc, xTics,yTics, bgColor) {
		this.maxVal = maxVal; this.minVal = minVal; this.inc = inc;
		this.boxHt = ht; this.boxWid = wid;
		
		var c = this.ctx;
		c.fillStyle = bgColor; // plot area body color
		c.fillRect(0,0,wid,ht);
		c.fillStyle = 'black';  // axis label color
		c.fillText(this.name, 15,15);
		
		// grid lines
		this.makeHorGridLines(maxVal, minVal, inc, yTics);
		this.makeVertGridLines(24, 0, 1, xTics); // hourly grid
		
		//set initial points
		this.endPts = [{x:0,y:0},{x:0,y:0},{x:0,y:0},{x:0,y:0}]; // last points plotted
		
		// legend space
		this.lgndTop = 10; 
		this.lgndLft = wid+30;
		this.lgndWid = 55;
		this.lgndHt = 60; 
		this.lgndSpacing = 15;
		this.lgndDefault = 5;
		this.lgndLastUsed = 5;
		c.fillStyle = 'white';
		c.fillRect (this.lgndLft, this.lgndTop, this.lgndWid, this.lgndHt);
};
	 
Chart.prototype.makeVertGridLines = function (maxVal, minVal, inc, tics) {
		var c = this.ctx,
				range = maxVal-minVal,
				nVerLines = (range/inc), 
				hSpace = this.boxWid/(range/inc),
				nTics = (inc/tics) * (nVerLines),
				tSpace = this.boxWid/(range/tics);
		c.lineWidth = 0.5;
				
		c.strokeStyle = 'black';
		for (var i=nTics; i>0; i--) {
			var startAt = i*tSpace;
			c.beginPath();
			c.moveTo(startAt-1,this.boxHt-2);
			c.lineTo(startAt-1,this.boxHt+2);
			c.stroke();
			c.closePath();
		};				
		
		c.strokeStyle = 'gray';
		for (var i=0; i<=nVerLines; i++) {
			c.beginPath();
			var startAt = i*hSpace;
			c.moveTo(startAt,0);
			c.lineTo(startAt,this.boxHt+5);
			c.stroke();
			c.closePath();
			c.fillText('|',startAt-2, this.boxHt+4);
			c.fillText(i, startAt-4,this.boxHt+18);
		};
};

Chart.prototype.makeHorGridLines = function (maxVal, minVal, inc, tics) {
		var c = this.ctx,
				range = maxVal-minVal,
				nHorLines = (range/inc), 
				vSpace = this.boxHt/(range/inc),
				nTics = (inc/tics) * (nHorLines),
				tSpace = this.boxHt/(range/tics);
		c.lineWidth = 0.5;
				
		for (var i=nTics; i>0; i--) {
			var startAt = i*tSpace;
			c.fillText('-', this.boxWid,startAt+2);
		};				
		
		c.strokeStyle = 'gray'; // grid color
		for (var i=nHorLines; i>0; i--) {
			c.beginPath();
			var startAt = i*vSpace;
			c.moveTo(0,startAt);
			c.lineTo(this.boxWid,startAt);
			c.stroke();
			c.closePath();
			c.fillText('_', this.boxWid-2, startAt-2);
			c.fillText(maxVal-(inc*i), this.boxWid+7, startAt+3);
		};
};	

Chart.prototype.colors = ['#be1e2d', '#666699', '#92d5ea', '#ee8310']; //plot line colors
	
Chart.prototype.adjX = function (x) {return x*(this.boxWid/24);};
Chart.prototype.adjY = function (y) {
	var range = this.maxVal-this.minVal;
	var px = y*(this.boxHt/range);
	var adj = this.boxHt/(range/this.minVal);
	//console.log('y='+y+'; range='+range+'; px='+px+'; adj='+adj)
	return this.boxHt-px+adj;
};	

Chart.prototype.plotPoint = function (l/*legend*/, x/*time*/, y/*value*/, i/*item index*/) {
		var c = this.ctx, 
				endPt = this.endPts[i]; // fetch last point plotted
		c.lineWidth = 1.0;
		c.strokeStyle = this.colors[i]; // plot color
		c.beginPath();

		c.moveTo(this.adjX(endPt.x), this.adjY(endPt.y));
		c.lineTo(this.adjX(x), this.adjY(y));
		c.stroke();
		c.closePath();
		this.endPts[i].x = x; // save for next time
		this.endPts[i].y = y; 
		
		// add legend
		c.fillStyle = this.colors[i];
		this.lgndLastUsed += this.lgndSpacing;
		c.fillText(l, this.lgndLft+5, this.lgndLastUsed)
};	
	//--//
	
function CCchart (id, name, width, height, bgColor) {
	Chart.call(this, id, name, width, height, bgColor);
};
CCchart.prototype = inherit(Chart.prototype);
CCchart.prototype.constructor = CCchart;

CCchart.prototype.plot = function (rpt, func) {
	//console.log('plotting CC')
  var ports = rpt.devstatus.ports;
  for(item in ports) {
  	if (ports[item].Dev == 'CC') {
			//console.log('found '+ports[item].Dev+' data');
    	var pvVolts = ports[item].In_V;
     	var pvAmps = ports[item].In_I;
     	var ccVolts = ports[item].Batt_V;
     	var ccAmps = ports[item].Out_I;

  		if (func) func(ports[item]);
     };
  };
	var parts = rpt.time.split(':');
	var theTime = parseFloat(parts[0])+(parseFloat(parts[1])+(parseFloat(parts[2])/60.0))/60.0;	
	this.lgndLastUsed = this.lgndDefault;
  this.plotPoint('PV Volts', theTime, pvVolts,0);
  this.plotPoint('PV Amps', theTime, pvAmps,1);
  this.plotPoint('CC Volts', theTime, ccVolts,2);
  this.plotPoint('CC Amps', theTime, ccAmps,3);
  
};

//--//
function DCchart (id, name, width, height, bgColor) {
	Chart.call(this, id, name, width, height, bgColor);
};
DCchart.prototype = inherit(Chart.prototype);
DCchart.prototype.constructor = DCchart;

DCchart.prototype.plot = function (rpt, func) {
	//console.log('plotting DC')
  var ports = rpt.devstatus.ports;
  for(item in ports) {
  	if (ports[item].Dev == 'FNDC') {
			//console.log('found '+ports[item].Dev+' data');
    	var shuntAI = ports[item].Shunt_A_I;
     	var shuntBI = ports[item].Shunt_B_I;

  		if (func) func(ports[item]);
    };
  };
	var parts = rpt.time.split(':');
	var theTime = parseFloat(parts[0])+(parseFloat(parts[1])+(parseFloat(parts[2])/60.0))/60.0;	
	this.lgndLastUsed = this.lgndDefault;
  this.plotPoint('Chg Amps', theTime, shuntAI, 0);
  this.plotPoint('Inv Amps', theTime, shuntBI, 1);
  this.plotPoint('Batt Net', theTime, (shuntAI+shuntBI), 2);
};

//--//
function FXchart (id, name,port, width, height, bgColor) {
	Chart.call(this, id, name, width, height, bgColor);
	this.port = port;
};
FXchart.prototype = inherit(Chart.prototype);
FXchart.prototype.constructor = FXchart;

FXchart.prototype.plot = function (rpt, func) {
	//console.log('plotting FX for port #'+this.port)
  var ports = rpt.devstatus.ports;
  for(item in ports) {
  	if (ports[item].Dev == 'FX' && ports[item].Port == this.port) {
			//console.log('found '+ports[item].Dev+' data');
    	var invI = parseFloat(ports[item].Inv_I);
     	var chgI = parseFloat(ports[item].Chg_I);
     	var buyI = parseFloat(ports[item].Buy_I);
     	var sellI = parseFloat(ports[item].Sell_I);
     	var vacIn = parseFloat(ports[item].VAC_in);
     	var vacOut = parseFloat(ports[item].VAC_out);

  		if (func) func(ports[item]);
    };
  };
  var invW = vacOut * invI;
  var chgW = vacIn * chgI;
  var lineW = (vacIn*buyI - vacOut*sellI);

	var parts = rpt.time.split(':');
	var theTime = parseFloat(parts[0])+(parseFloat(parts[1])+(parseFloat(parts[2])/60.0))/60.0;	
	this.lgndLastUsed = this.lgndDefault;
  this.plotPoint('Line Watts', theTime, lineW, 0);
  this.plotPoint('Inv Watts', theTime, invW, 1);
  this.plotPoint('Chg Watts', theTime, chgW, 2);
};
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //

OBP = {
		oldJsonPath: 'OBdata.txt',
		oldXmlPath: 'OBconfig.txt',
  	srvrPath: 'mate3Srvr.php',
  	fileIntvl: 60000.0,	// file contains 1 minute data in seconds
  	jsonIntvl: 5000.0,	// real time data retreival interval in seconds
  	xmlIntvl: 800000.0,	// config data retreival interval in seconds
  	kWhIntvl: 0.0,			// set to one of above as appropriate
  	getRealTime: false,	// enable real time updating
  	
  	xmlDoc: "",
  	jsonData: "",
  	_inverters: null,
  	_globalInverter: null,
		
	init: function () {
		// adjust menu and operation for server location
		$('.homesites').hide();
		//console.log('hostName = '+location.hostname);		
		switch (location.hostname) {
			case 'aspen': 
			case 'www.flos-inc.local': 
			case '192.168.0.7': 
			case 'localhost': $('#devSite').show(); OBP.getRealTime = true; break;
			case 'www.flos-inc.com': $('#webSite').show(); OBP.getRealTime = false; break;
		}
		$('.no-touch #chartsTbl').removeClass('tabData'); // charts sometimes visible
		$('.tabData').hide();	// hide all tabular data for now
		makeNav();	// activate the navigation menu

	  // prepare for menu selections
		$('nav div.outback a').bind('click',null,function (e) { 
  		var theId = e.target.id;
  		$('nav a').removeClass('selected');
  		$('nav a#'+theId).addClass('selected');
    	$('#data .tabData').each( function(index) {
    	 	if (this.id == (theId+'Tbl')) {
    	 	  $(this).show();	//.removeClass('selected');
    	 	} else {
   	  	  $(this).hide(); //.addClass('selected');
    		}
    	});
  	});    
		$('nav a#meters').triggerHandler('click');

		//prepare for accumulated values
   	OBP.store = {
			'buyKwh': 0.0,'sellKwh': 0.0,'loadKwh': 0.0,'chgKwh': 0.0,
			'fxBuyAmp': 0.0, 'fxSellAmp': 0.0, 'fxInvAmp': 0.0, 'fxChgAmp': 0.0,
			'solarKwh':0.0,'invKwh':0.0,
			'pvKwh': 0.0,'chgrKwh':0.0,'pkPvAmp':0.0,'pkChgrAmp':0.0,
		};

		// prepare file-loading progress bar
		//pbar.init( 2,2, 200,34 );
		
		OBP.prepareChartSpace();

		// get initial data to display
		OBP.fetchPastData();	// historic realtime data in JSON format

		// start normal retreival cycle
		OBP.hasReset = false; // controls midnight reset
		if (OBP.getRealTime) {
    	setInterval( "OBP.fetchCrntData()", OBP.jsonIntvl );
    	setInterval( "OBP.updateStatus()", OBP.xmlIntvl );
    };
	},
	
	prepareChartSpace: function () {
  	// prepare chart spaces for plotting
  	var chartBg = 'hsla(60, 100%, 85%, 0.5)', //fully saturated light yellow, 50% opague
  			boxBg = 'hsla(0, 0%, 95%, 1.0)'; //pale gray, non-opague
  	OBP.fx1 = new FXchart('fx1','Inverter-Charger #1',1, 570, 220, boxBg);
  	OBP.fx1.makeBox(0,0, 480,200, 1400,-1400,200, .25,50, chartBg);
  	OBP.fx2 = new FXchart('fx2','Inverter-Charger #2',2, 570, 220, boxBg);
  	OBP.fx2.makeBox(0,0, 480,200, 1400,-1400,200, .25,50, chartBg);
  	OBP.dc = new DCchart('fndc','Battery Monitor', 570, 180, boxBg);
  	OBP.dc.makeBox(0,0, 480,160, 80,-80,20, .25,5, chartBg);
  	OBP.cc = new CCchart('cc','PV Charge Controller', 570, 180, boxBg);
  	OBP.cc.makeBox(0,0, 480,160, 80,0,10, .25,2, chartBg);
	},
	
	reDraw: function () {
		OBP.fetchCrntData();
  	OBP.updateDisplay();
	},
	
	setProgLimit: function (n) {
		this.limit = n;
	},
	showProgress: function (n) {
		var fraction = parseFloat(n) / parseFloat(this.limit);
		var prog = (fraction * 100.0).toFixed(0);
		$('progress > p').html(prog+'% complete');
	},
	
	fetchPastData: function () {
		$('progress').show();
		OBP.clearDataGrps();
		$.ajax({url: OBP.oldJsonPath,
						dataType: 'text',
						context: this,
						success: function(manyRpts) {
							var rptArray = manyRpts.split('\n'),
									nJson = rptArray.length,
									json,
									now = new Date();						 		

							theTime = (now.getHours()+':'+now.getMinutes()+':'+now.getSeconds());
    					$('span.asofJsonTime').html(theTime);

							// each rpt in array contains a full set of data for all devices
							//pbar.setProgLimit(nJson-1);  // prepare progress bar
							OBP.setProgLimit(nJson-1);	// prepare progress
							for(var n=0; n<nJson-1; n++) {
								//pbar.draw(n);	// display progress so far
								OBP.showProgress(n);	// display progress so far
								try { json = JSON.parse(rptArray[n]); }
								catch (e) { console.log (e+'; \n'+jsonArray(n)); continue; };
								OBP.jsonData = json;	// just in case
							  OBP.updatePlots (this, json, true);
							};
							
							// update tables using last file report only
							OBP.kWhIntvl = OBP.fileIntvl;
    					Meters.update(json);
    					Status.update(json);
    					Mode.updateFromJSON(json);
							$('progress').hide();

							// fetch most recent 'static' data in XML format
							OBP.updateStatus();	
						},
						error : function () {
							console.log('fetching old plot data from '+this.oldJsonPath+' failed');
						}
		});
	},
	
	fetchCrntData: function () {			
    $.ajax({url: this.srvrPath+'?mode=getCrntData', 
						dataType: 'json', 
						context: this,
						success: function (json) {												 		
							var now = new Date(),
									theHr = now.getHours(),
									theMin = now.getMinutes(),
									theSec = now.getSeconds(),
									hasReset = OBP.hasReset;
									
							// determine if time to reset charts for new day		
							json.time = (theHr+':'+theMin+':'+theSec);
    					$('span.asofJsonTime').html(' @ '+json.time);
    					if (!hasReset && ((theHr == 0)&&(theMin == 0))) {
								OBP.prepareChartSpace();
								OBP.hasReset = true;
							}
							else if ((theHr == 23) && hasReset ) { 
								OBP.hasReset = false;
							}
							
						  OBP.jsonData = json;
							OBP.updatePlots (this, json, false);						  

							OBP.kWhIntvl = OBP.jsonIntvl;			
    					Meters.update(json);
    					Status.update(json);
    					Mode.updateFromJSON(json);
						},
						error : function () {
							console.log('AJAX() FAILURE: fetching current data from '+this.srvrPath);
						}
		});
	},
	
 	updateStatus: function() {
		if (OBP.getRealTime) {
			var theUrl = this.srvrPath+'?mode=getCrntStatus';
			var theType = 'xml';
		}
		else {
			var theUrl = OBP.oldXmlPath;
			var theType = 'text'
		}
		this.clearDataGrps();
    $.ajax({url: theUrl, 
						dataType: theType, 
						context: this,
						success: function (xml) {
							try { var trialStr = xml.replace(/\\\\/g, ''); xml = trialStr; }							
							catch	(e) { console.log("removing '\\' from XML failed"); }						
    					this.xmlDoc = xml;
							OBP.kWhIntvl = OBP.xmlIntvl;
    					Mode.updateFromXML(xml);
    					Config.update(xml);
    					Setup.update(xml);

							var now = new Date();						 		
							xmltime = (now.getHours()+':'+now.getMinutes()+':'+now.getSeconds());
							$('span.asofXmlTime').html(' @ '+xmltime);
						},
						error : function () {
							console.log("fetching 'static' config from "+theUrl+" failed");
						} 
		});
  },
  
	updatePlots: function (thisObj, rpt, doStore) {
		thisObj.cc.plot(rpt, function (data) {
			if (doStore) {
				OBP.store.pvKwh += ((data.In_V * data.In_I)/1000.0)/60.01;	
				OBP.store.chgrKwh += ((data.Batt_V * data.Out_I)/1000.0)/60.01;	
				OBP.store.pkPvAmp = Math.max(OBP.store.pkPvAmp, data.In_I);
				OBP.store.pkChgrAmp = Math.max(OBP.store.pkChgrAmp, data.Out_I);
			}
		});
		thisObj.dc.plot(rpt, function (data) {
			if (doStore) {
				OBP.store.solarKwh += ((data.Batt_V * data.Shunt_A_I)/1000.0)/60.01;	
				OBP.store.invKwh += ((data.Batt_V * data.Shunt_B_I)/1000.0)/60.01;	
			}
		});
		thisObj.fx1.plot(rpt, function (data) {
			if (doStore) {
				OBP.store.buyKwh += ((data.VAC_in * data.Buy_I)/1000.0)/60.01;	
				OBP.store.sellKwh += ((data.VAC_out * data.Sell_I)/1000.0)/60.01;	
				OBP.store.loadKwh += ((data.VAC_out * data.Inv_I)/1000.0)/60.01;	
				OBP.store.chgKwh += ((data.VAC_in * data.Chg_I)/1000.0)/60.01;	
			}
		});
		thisObj.fx2.plot(rpt, function (data) {
			if (doStore) {
				OBP.store.buyKwh += ((data.VAC_in * data.Buy_I)/1000.0)/60.01;	
				OBP.store.sellKwh += ((data.VAC_out * data.Sell_I)/1000.0)/60.01;	
				OBP.store.loadKwh += ((data.VAC_out * data.Inv_I)/1000.0)/60.01;	
				OBP.store.chgKwh += ((data.VAC_in * data.Chg_I)/1000.0)/60.01;	
			}
		});
	},
	
	clearDataGrps: function () {
		$('div.dataGrp').html('');
	},
	
  getDevices: function(deviceName) {
    var ports = OBP.jsonData.devstatus.ports;
    var devices = []
    for(name in ports) {
      if (ports[name].Dev == deviceName) {
        devices.push(ports[name]);
      };
    }
    return devices;
  },
  
  sortInverters: function(a, b) {
    return ((a.Port < b.Port) ? -1 : ((a.Port > b.Port) ? 1 : 0));
  },
  
  getInverters: function() {
      this._inverters = this.getDevices('FX');
    return this._inverters;
  },
  
  getGlobalInverter: function() {
      this._globalInverter =  this.getInverters().sort(this.sortInverters); //[0];
    return this._globalInverter;
  },
  
	updateHTMLFromXMLAttribute: function (theDiv, key, value, xml) {
  	var nodes = $(xml).find(value),
				theVal = nodes.first().attr("Type");
    
		if (typeof theVal == 'undefined') theVal = 'n/a';
		
    var $key = $('div#'+theDiv+'Tbl > table '+key);
    $key.html(theVal);
	},
	
	updateHTMLFromXMLElement: function (theDiv, key, value, xml) {
  	var nodes = $(xml).find(value),
				theVal = nodes.first().text();
    
    var $key = $('div#'+theDiv+'Tbl > table '+key);
    $key.html(theVal);

    if (key == "#current_time") {
     	var timeStr = (nodes.first().text()).substr(-5,5);
			OBP.timeStamp = timeStr;
		}
	},
	
  updateHTMLFromJSONMap: function(theDiv, map, dataSource) {
		if ($.isArray(dataSource)) {
			this.doMap(theDiv, map, dataSource[0]);
		}
		else 
			this.doMap(theDiv, map, dataSource);
  },
  
  doMap: function (theDiv, map, dataSource) {
  	var crntSrce='';
    for( item in map ) {
    	var theSrce = dataSource.Dev;
    	if (crntSrce != theSrce){
		    crntSrce = theSrce;
				if ((theDiv != 'meters') && (theDiv != 'status') && (theDiv != 'mode'))
					$('div#'+theDiv+'Tbl > table').append('<tr><th colspan="2" class="secLbl">--- '+crntSrce+' ---</th></tr>');
			};
    	OBP.updateHTMLFromJSONItem(theDiv, item.substr(1), dataSource, map[item]);
    };
	},
	
	updateHTMLFromJSONItem: function (theDiv, key, source, entry) {
		if ($.isArray(entry)) {
			var rule = entry,
					theVal;
			if (rule[2] == null)  {
				theVal = OBP.performCalc(source[rule[0]], rule[1], source[rule[2]], rule[3]);
			}
			else if (typeof rule[2] == 'number' ) {
				theVal = OBP.performCalc(source[rule[0]], rule[1], rule[2], rule[3]);
			}
			else if (rule[2].substr(0,3) == 'OBP') {
			  var theStore = rule[2].substr(10,999);
			  var operand = OBP.store[theStore];
				theVal = OBP.performCalc(source[rule[0]], rule[1], operand, rule[3]);
			}
			else  {
				theVal = OBP.performCalc(source[rule[0]], rule[1], source[rule[2]], rule[3]);
			}
		}
		else if (source === null) {
			theVal = entry;	
		}
		else {
			var data = source[entry]; 
			if ($.isArray(data)) {
				theVal = data.join(' | ');
				if (key.indexOf('@') >= 0) {
					key2 = key.substr(0,key.length-1)
					key = key2+source['Port'];
				}
			}
			else if (typeof data == 'boolean') {
				if (theVal == true) 
					theVal = 'true'; 
				else 
					theVal = 'false';
			}	
			else {
				theVal = data;
				if (key.indexOf('@') >= 0) {
					key2 = key.substr(0,key.length-1)
					key = key2+source['Port'];
				}
			}
		}

    $('#'+key).html(theVal);
	},
  
  performCalc: function (a, op, b, opt) {
		//console.log ('a='+a+'; op='+op+'; b='+b+'; opt='+opt);  
  	var A = parseFloat(a), 
				B = parseFloat(b),
				rslt = 0.0,
				theName = '',
				theValue;
  	if (op) { // not null or missing
  		switch (op){
				case '+': rslt = (A + B); break;
				case '-': rslt = (A - B); break;
				case '*': rslt = (A * B); break;
				case '/': rslt = (A / B); break;
				default: rslt = 'error';	
			};
		};
		if (opt != undefined) {
			if (opt['accumKwh'] != undefined)  {
				theName = opt['accumKwh'];
				theValue = OBP.store[theName]
				if ((isNaN(theValue)) || (theValue == undefined) ){
					console.log ('bad data ('+theValue+') stored for: '+theName+'; set to zero');
					OBP.store[theName] = 0.0;
				}
				var kw = rslt / 1000.0,
						intvlSecs = OBP.kWhIntvl / 1000.0,
						divisor = 3600.0/intvlSecs,
						addKwh = (kw / divisor)
						ttlKwh = OBP.store[theName] + addKwh;
				rslt = ttlKwh;
				if (isNaN(rslt) || (rslt == undefined)) 
					console.log ('bad data: a='+a+'; op='+op+'; b='+b+'; opt='+opt);
				else
					OBP.store[theName] = rslt;
			}
			if (opt['peak'] != undefined){
				theName = opt['peak'];
				OBP.store[theName] = Math.max(OBP.store[theName], A);
				rslt = OBP.store[theName];
			}
			if (opt['accumVal'] != undefined ) {
				theName = opt['accumVal'];
				var temp = OBP.store[theName] + rslt;
        rslt = temp;	
				OBP.store[theName] = rslt;
			}
		};
		return rslt.toFixed(1);
	},
};

//-----------------------------
Meters = {
	htmlToDataSourceMap: {
    "inverter" : {   
      "#out_volts" 		: "VAC_out",
      '#load_amps'		: ['Inv_I','+',0.0, {'accumVal':'fxInvAmp'}],
      '#load_watts'		: ['VAC_out','*','OBP.store.fxInvAmp'],
      '#load_kwh'			: ['VAC_out','*','Inv_I', {'accumKwh':'loadKwh'}],

      '#charge_amps'	: ['Chg_I','+',0.0, {'accumVal':'fxChgAmp'}],
      '#charge_watts'	: ['VAC_in','*','OBP.store.fxChgAmp'],
      '#charge_kwh'		: ['VAC_in','*','Chg_I', {'accumKwh':'chgKwh'}],

      "#line_volts" 	: "VAC_in",
      '#sell_amps'		: ['Sell_I','+',0.0, {'accumVal':'fxSellAmp'}],
      '#sell_watts'		: ['VAC_out','*','OBP.store.fxSellAmp'],
      '#sell_kwh'			: ['VAC_out','*','Sell_I', {'accumKwh':'sellKwh'}],

      '#buy_amps'			: ['Buy_I','+',0.0, {'accumVal':'fxBuyAmp'}],
      '#buy_watts'		: ['VAC_in','*','OBP.store.fxBuyAmp'],
      '#buy_kwh'			: ['VAC_in','*','Buy_I', {'accumKwh':'buyKwh'}],

      '#ac_mode@'			: "AC_mode",
      '#inv_mode@'		: "INV_mode",
      '#warning@'			: "Warn",
      '#error@'				: "Error",
      '#fx_aux@'			: "AUX",
    },
    "fndc" : {
    	'#shunts'			: "Enabled",
      "#solar_amps" : "Shunt_A_I",
      '#solar_watts': ['Batt_V','*','Shunt_A_I'],
      '#solar_kwh'	: ['Batt_V','*','Shunt_A_I', {'accumKwh':'solarKwh'}],
      //'#solar_kwh'	: 'Shunt_A_kWh',
      "#inv_amps" 	: "Shunt_B_I",
      '#inv_watts'	: ['Batt_V','*','Shunt_B_I'],
      '#inv_kwh'		: ['Batt_V','*','Shunt_B_I', {'accumKwh':'invKwh'}],
      //'#inv_kwh'		: 'Shunt_B_kWh',
      "#btry_volts" : "Batt_V",
      '#btry_amps'	: ['Shunt_A_I','+','Shunt_B_I'],
      '#soc'				:	"SOC",
      '#socMin'			:	"Min_SOC",
      '#sinceFull'	: "Days_since_full",
      '#paramsMet'	: "CHG_parms_met",
      "#btry_temp" 	: "Batt_temp",
      '#dcAux_mode'	: 'Aux_mode',
      '#dcAux'			: "AUX",
      
      "#channel_a_accumulated_ah" : "Shunt_A_AH",
      "#channel_b_accumulated_ah" : "Shunt_B_AH",
      "#total_accumulated_ah" 		: "In_AH_today",
      "#channel_a_accumulated_kwh": "Shunt_A_kWh",
      "#channel_b_accumulated_kwh": "Shunt_B_kWh",
      "#total_accumulated_kwh" 		: "In_kWh_today",
      "#todays_net_input_ah" 			: "In_AH_today",
      "#todays_net_input_kwh" 		: "In_kWh_today",
      "#todays_net_output_ah" 		: "Out_AH_today",
      "#todays_net_output_kwh" 		: "Out_kWh_today",
    },
    "charge_controller" : {   
      "#pv_volts"		: "In_V",
      "#pv_amps" 		: "In_I",
      "#pv_watts"		: ['In_V','*','In_I'],
      '#pv_kwh'			: ['In_V','*','In_I', {'accumKwh':'pvKwh'}],
      '#pv_pkAmp'		: ['In_I',null,null,{'peak':'pkPvAmp'}],
      '#chgr_volts'	: "Batt_V",
      "#chgr_amps" 	: "Out_I",
      "#chgr_watts"	: ['Batt_V','*','Out_I'],
      "#chgr_kwh" 	: ['Batt_V','*','Out_I', {'accumKwh':'chgrKwh'}],
      //"#chgr_kwh" 	: "Out_kWh",
      '#chgr_pkAmp'	: ['Out_I',null,null,{'peak':'pkChgrAmp'}],
			'#chgr_error'	: 'Error',
      '#chgr_mode'	: 'CC_mode',
      '#ccAux_mode'	: 'Aux_mode',
      '#ccAux'			: "AUX",
      
      "#daily_ah" 	: "Out_AH",
    }        
	},
  
	update: function(json) {
    var globalInverter = OBP.getGlobalInverter();
    var chargeController = OBP.getDevices('CC')[0];
    var fndc = OBP.getDevices('FNDC')[0];
    
    var processArray = [
      {"map" : this.htmlToDataSourceMap.inverter, "dataSource" : globalInverter[0]},
      {"map" : this.htmlToDataSourceMap.inverter, "dataSource" : globalInverter[1]},
      {"map" : this.htmlToDataSourceMap.charge_controller, "dataSource" : chargeController},
      {"map" : this.htmlToDataSourceMap.fndc, "dataSource" : fndc}
    ];
    
    var firstFx = true;
    for( index in processArray ) {
      if (firstFx && ( processArray[index].dataSource.Dev == 'FX' )) {
      	// initialize to get totals for both inverters
				OBP.store['fxBuyAmp'] = 0.0; 
				OBP.store['fxSellAmp'] = 0.0;
				OBP.store['fxInvAmp'] = 0.0;
				OBP.store['fxChgAmp'] = 0.0;
				firstFx = false;
			} 
      OBP.updateHTMLFromJSONMap('meters', processArray[index].map, processArray[index].dataSource);
    }
  },
};

//-----------------------------
Status = {
	htmlToDataSourceMap: {
    "FNDC" : {
      "#days_since_full" 		: "Days_since_full",
      "#battery_charge" 		: "SOC",
      "#todays_minimum_charge" : "Min_SOC",
      "#net_battery_ah" 		: "Net_CFC_AH",
      "#net_battery_kwh" 		: "Net_CFC_kWh",  
      "#channel_a_current" 	: "Shunt_A_I",
      "#channel_b_current" 	: "Shunt_B_I",
      "#total_current"			: ['Shunt_A_I','+','Shunt_B_I'],
    }
	},

	update: function(json) {
//console.log('posting status');	
    var fndc = OBP.getDevices('FNDC')[0];
    
    var processArray = [
      {"map" : this.htmlToDataSourceMap.FNDC, "dataSource" : fndc}
    ];
    
    for( index in processArray ) {
      OBP.updateHTMLFromJSONMap('status', processArray[index].map, processArray[index].dataSource);
    }
  },
};

//-----------------------------
Config = {
	htmlToDataSourceMap: {
      "#system_name" : "Sys_Name",
      "#system_type" : "System[Type]",
      "#nominal_voltage" : "Device[Type='FX'] Voltage",
      "#pv_wattage" : "PV_Size_Watts",
      "#battery_capacity" : "Battery_AH_Capacity",
      "#generator_kw" : "Generator_KW",
      "#max_inverter_output" : "Max_Inverter_Output_KW",
      "#max_charger_output" : "Max_Charger_Output_KW",
      "#current_time" : "Time_Stamp"     
	},
  
	update: function(xml) {   
//console.log('posting config');	
    $.each(this.htmlToDataSourceMap, function(key, value) {
      if (key == "#system_type") {
        OBP.updateHTMLFromXMLAttribute('config', key, value, xml);
      } else {        
        OBP.updateHTMLFromXMLElement('config', key, value, xml);
      }
    });
  },
};

//-----------------------------
Mode = {
	inverters: [],
	
	htmlToDataSourceMap: {
    "xml": {
      "mode_attributes" : {
        "#inverter_mode_global" : "Port[Num='1'] > Device[Type='FX'] > Inverter",
        "#charger_mode" : "Port[Num='1'] > Device[Type='FX'] > Charger",
        "#modetab_grid_use_mode": "System_Config Grid_Use",
        "#modetab_hbx_mode": "System_Config High_Battery_Transfer",           
        "#modetab_ags_mode": "System_Config Advance_Generator_Start",  
        "#inverter_aux_output_mode": "Port[Num='1'] > Device[Type='FX'] AUX_Output",   
        "#charge_controller_aux_operation_mode": "Device[Type='CC'] Model[Type='FM'] MPPT",        
      },
      "element_values" : {
        "#nominal_voltage" : "Port[Num='1'] > Device[Type='FX'] Voltage",
        "#pv_wattage" : "PV_Size_Watts",
        "#battery_capacity" : "System_Config > Battery_AH_Capacity",
        "#generator_kw" : "System_Config > Generator_KW",        
        "#inverter_aux_operation_mode": "Port[Num='1'] > Device[Type='FX'] AUX_Output > Operation_Mode",     
        "#max_inverter_output" : "Max_Inverter_Output_KW",
        "#max_charger_output" : "Max_Charger_Output_KW",
        "#current_time" : "Time_Stamp"        
      }    
    },
    "json": {
      "inverter" : {
        "#inverter_mode_port" : "INV_mode",
        "#inverter_aux_output_state": "AUX",
        "#ac_input_global": "AC_mode",
        "#ac_input_port": "AC_mode",   
      },
      "charge_controller": {
        "#charge_controller_mode": "CC_mode",
        "#charge_controller_aux_output_state": "AUX",
        "#charge_controller_aux_output_mode": "Aux_mode"
      },
      "fndc": {
        "#fndc_aux_state": "AUX",
        "#fndc_aux_output_mode": "Aux_mode"
      } 
    }
	},
	
	updateFromXML: function(xml) {
//console.log('posting mode from xml');	
    $.each(this.htmlToDataSourceMap.xml.mode_attributes, function(key, value) { 
        OBP.updateHTMLFromXMLAttribute('mode', key, value, xml);
    });    

    $.each(this.htmlToDataSourceMap.xml.element_values, function(key, value) {       
        OBP.updateHTMLFromXMLElement('mode', key, value, xml);
    });
  },

	updateFromJSON: function(json) {
//console.log('posting mode from json');	
    var globalInverter = OBP.getGlobalInverter();
    var chargeController = OBP.getDevices('CC')[0];
    var fndc = OBP.getDevices('FNDC')[0];
    
    var processArray = [
      {"map" : this.htmlToDataSourceMap.json.inverter, "dataSource" : globalInverter},
      {"map" : this.htmlToDataSourceMap.json.charge_controller, "dataSource" : chargeController},
      {"map" : this.htmlToDataSourceMap.json.fndc, "dataSource" : fndc}
    ];
    
    for( index in processArray ) {
      OBP.updateHTMLFromJSONMap('mode', processArray[index].map, processArray[index].dataSource);
    }
  },
    
};

//-----------------------------
Setup = {
	htmlToDataSourceMap: {
    "mode_attributes" : {
      // inverter
      "#setup_inverter_mode" : "Port[Num='1'] > Device[Type='FX'] > Inverter", 
      "#stack_mode" : "Port[Num='1'] > Device[Type='FX'] > Stack", 
      "#charge_mode" : "Port[Num='1'] > Device[Type='FX'] > Charger", 
      "#gridtie_mode" : "Port[Num='1'] > Device[Type='FX'] > Grid_tie", 
      "#ags_mode" : "Advance_Generator_Start", 
      "#two_minute_start_mode" : "Advance_Generator_Start > Two_Min_Voltage_Start", 
      "#two_hour_start_mode" : "Advance_Generator_Start > Two_Hour_Voltage_Start", 
      "#24_hour_start_mode" : "Advance_Generator_Start > Twenty_Four_Hour_Voltage_Start", 
      "#hbx_mode" : "High_Battery_Transfer",
      "#aux_output_mode" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output",
      // charge_controller
      "#cc_aux_output_mode" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output",
      // fndc
      "#controller_float" : "CC_Float_Coordination",
      "#fndc_charge" : "FNDC_Charge_Term_Control",
      "#fndc_sell" : "FNDC_Sell_Control"
    },  
    "element_values" : {
      // inverter
      "#pulse_length" : "Port[Num='1'] > Device[Type='FX'] > Inverter > Search > Pulse_Length",
      "#ac_load_threshold" : "Port[Num='1'] > Device[Type='FX'] > Inverter > Search > Amps",
      "#pulse_spacing" : "Port[Num='1'] > Device[Type='FX'] > Inverter > Search > Spacing",
      "#low_battery_cutout_voltage" : "Port[Num='1'] > Device[Type='FX'] > Inverter > Low_battery > Cut_Out_Voltage",
      "#restart_voltage" : "Port[Num='1'] > Device[Type='FX'] > Inverter > Low_battery > Cut_In_Voltage",
      "#ac_output" : "Port[Num='1'] > Device[Type='FX'] > Inverter > AC_Output_Voltage",
      "#master_power" : "Port[Num='1'] > Device[Type='FX'] > Stack > Master_Power_Save_Level",
      "#slave_power" : "Port[Num='1'] > Device[Type='FX'] > Stack > Slave_Power_Save_Level",
      "#absorb_voltage" : "Port[Num='1'] > Device[Type='FX'] > Charger > Absorb > Voltage",
      "#absorb_time" : "Port[Num='1'] > Device[Type='FX'] > Charger > Absorb > Time",
      "#float_voltage" : "Port[Num='1'] > Device[Type='FX'] > Charger > Float > Voltage",
      "#float_time" : "Port[Num='1'] > Device[Type='FX'] > Charger > Float > Time",
      "#eq_voltage" : "Port[Num='1'] > Device[Type='FX'] > Charger > EQ > Voltage",
      "#eq_time" : "Port[Num='1'] > Device[Type='FX'] > Charger > EQ > Time",
      "#refloat_voltage" : "Port[Num='1'] > Device[Type='FX'] > Charger > Re_Float > Voltage",
      "#charger_ac" : "Port[Num='1'] > Device[Type='FX'] > Charger > AC_Charger_Input_Limit",
      "#sell_voltage" : "Port[Num='1'] > Device[Type='FX'] > Grid_tie > Voltage",
      "#sell_window" : "Port[Num='1'] > Device[Type='FX'] > Grid_tie > Window",
      "#setup_fx_ac_input" : "Port[Num='1'] > Device[Type='FX'] > AC_Input_Select",
      "#ac1" : "Port[Num='1'] > Device[Type='FX'] > AC1_input",
      "#ac2" : "Port[Num='1'] > Device[Type='FX'] > AC2_input",
      "#connect_delay" : "Port[Num='1'] > Device[Type='FX'] > AC2_input > Connect_Delay",
      "#gen_support" : "Port[Num='1'] > Device[Type='FX'] > AC2_input > Gen_Support",
      "#ags_port" : "Advance_Generator_Start > Port",
      "#dc_generator" : "Advance_Generator_Start > DC_Generator",
      "#dc_generator_stop_voltage" : "Advance_Generator_Start > DC_Generator_Stop_Voltage",
      "#fault_time" : "Advance_Generator_Start > Fault_Time",
      "#cool_down" : "Advance_Generator_Start > Cooldown_Time",
      "#warmup_time" : "Advance_Generator_Start > Warmup_Time",
      "#two_minute_start_voltage" : "Advance_Generator_Start > Two_Min_Voltage_Start > Voltage",
      "#two_hour_start_voltage" : "Advance_Generator_Start > Two_Hour_Voltage_Start > Voltage", 
      "#24_hour_start_voltage" : "Advance_Generator_Start > Twenty_Four_Hour_Voltage_Start > Voltage", 
      // TODO: need to write a method to process date nodes in this format
      // <Quiet_Time Mode="disabled">
      //  <Weekday>
      //    <Start_Hour>0</Start_Hour>
      //    <Start_Min>0</Start_Min>
      //    <Stop_Hour>0</Stop_Hour>
      //    <Stop_Min>0</Stop_Min>
      //  </Weekday>
      //  <Weekend>
      //    <Start_Hour>0</Start_Hour>
      //    <Start_Min>0</Start_Min>
      //    <Stop_Hour>0</Stop_Hour>
      //    <Stop_Min>0</Stop_Min>
      //  </Weekend>
      // </Quiet_Time>      
      // "quiet_time_mode" : "Advance_Generator_Start > Quiet_Time", // Mode
      // "start_time" : "Advance_Generator_Start > ",
      // "stop_time" : "Advance_Generator_Start > ",
      // "weekend" : "Advance_Generator_Start > ",
      // "start_time" : "Advance_Generator_Start > ",
      // "stop_time" : "Advance_Generator_Start > ",
      "#low_voltage_connect_voltage" : "High_Battery_Transfer > Low_Voltage_Connect > Voltage",
      "#low_voltage_connect_delay" : "High_Battery_Transfer > Low_Voltage_Connect > Delay",
      "#high_voltage_disconnect_voltage" : "High_Battery_Transfer > High_Voltage_Disconnect > Voltage",
      "#high_voltage_disconnect_delay" : "High_Battery_Transfer > High_Voltage_Disconnect > Delay", 
      "#aux_output_operation_mode" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > Operation_Mode",
      "#on_setpoint_voltage_genalert" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Gen_Alert > On_Setpoint_Voltage",
      "#on_delay_genalert" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Gen_Alert > On_Delay",
      "#off_setpoint_voltage_genalert" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Gen_Alert > Off_Setpoint_Voltage",
      "#off_delay_genalert" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Gen_Alert > Off_Delay",
      "#on_setpoint_voltage_ventfan" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Vent_Fan > On_Setpoint_Voltage",
      "#off_delay_ventfan" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Vent_Fan > Off_Delay",
      "#on_setpoint_voltage_diversion" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Diversion > On_Setpoint_Voltage",
      "#off_delay_diversion" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Diversion > Off_Delay",
      "#off_voltage_loadshed" : "Port[Num='1'] > Device[Type='FX'] > AUX_Output > AUX_Loadshed_Off_Voltage",  
      // charge_controller
      "#cc_absorb_voltage" : "Device[Type='CC'] > Model[Type='FM'] Absorb > Voltage",
      "#cc_absorb_time" : "Device[Type='CC'] > Model[Type='FM'] Absorb > Time",
      "#end_amps" : "Device[Type='CC'] > Model[Type='FM'] Absorb > End_Amps",
      "#rebulk_voltage" : "Device[Type='CC'] > Model[Type='FM'] Rebulk_Voltage",
      "#cc_float_voltage" : "Device[Type='CC'] > Model[Type='FM'] Float_Voltage",
      "#cc_eq_voltage" : "Device[Type='CC'] > Model[Type='FM'] EQ > Voltage",
      "#cc_eq_time" : "Device[Type='CC'] > Model[Type='FM'] EQ > Time",
      "#auto_eq" : "Device[Type='CC'] > Model[Type='FM'] EQ > Auto_Interval",  
      "#cc_aux_output_status" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > Operation_Mode",
      "#logic_polarity" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > Polarity",
      "#disconnect_voltage_low_batt" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Low_Batt_Disconnect > Disconnect_Voltage",
      "#disconnect_delay_low_batt" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Low_Batt_Disconnect > Disconnect_Delay",
      "#reconnect_voltage_low_batt" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Low_Batt_Disconnect > Reconnect_Voltage",
      "#on_voltage_ventfan" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Vent_Fan_Voltage",
      "#hold_time_diversion" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Diversion > Hold_Time",
      "#delay_diversion" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Diversion > Delay",
      "#relative_volts_diversion" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Diversion > Relative_Voltage",
      "#hysteresis_volts_diversion" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Diversion > Hysteresis_Voltage",
      "#trigger_volts_pv_trigger" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_PV_Trigger > Trigger_Voltage",
      "#hold_time_pv_trigger" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_PV_Trigger > Hold_Time",
      "#threshold_volts_night_light" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Nite_Light > Threshold_Voltage",
      "#hysteresis_on_time_night_light" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Nite_Light > On_Hyst_Time",
      "#hysteresis_off_time_night_light" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Nite_Light > Off_Hyst_Time",
      "#on_time_night_light" : "Device[Type='CC'] > Model[Type='FM'] > AUX_Output > AUX_Nite_Light > On_Hours",
      // fndc 
      "#battery_ah" : "Device[Type='FNDC'] > Battery_AH_Capacity",
      "#shunt_a_status" : "Device[Type='FNDC'] > Shunt_A",
      "#shunt_b_status" : "Device[Type='FNDC'] > Shunt_B",
      "#shunt_c_status" : "Device[Type='FNDC'] > Shunt_C",
      "#return_amps" : "Device[Type='FNDC'] > Parms_Met > Return_Amps",
      "#full_voltage" : "Device[Type='FNDC'] > Parms_Met > Full_Volts",
      "#time_at" : "Device[Type='FNDC'] > Parms_Met > Time",
      "#charge_efficiency" : "Device[Type='FNDC'] > Charge_Factor",
      "#aux_relay_mode" : "Device[Type='FNDC'] > AUX_Relay > Mode",
      "#fndc_logic_polarity" : "Device[Type='FNDC'] > AUX_Relay > Logic",
      "#voltage_high" : "Device[Type='FNDC'] > AUX_Relay > Volts_High",
      "#voltage_low" : "Device[Type='FNDC'] > AUX_Relay > Volts_Low",
      "#soc_high" : "Device[Type='FNDC'] > AUX_Relay > SOC_High",
      "#soc_low" : "Device[Type='FNDC'] > AUX_Relay > SOC_Low",
      "#high_limit" : "Device[Type='FNDC'] > AUX_Relay > Delay_High",
      "#low_limit" : "Device[Type='FNDC'] > AUX_Relay > Delay_Low",
      "#low_soc_warning" : "Low_SOC_Warning_Percentage",
      "#low_soc_error" : "Low_SOC_Error_Percentage"      
    }
	},
  
	update: function(xml) {   
//console.log('posting setup from xml');	 	
    $.each(this.htmlToDataSourceMap.mode_attributes, function(key, value) { 
        OBP.updateHTMLFromXMLAttribute('setup', key, value, xml);
    });    
    $.each(this.htmlToDataSourceMap.element_values, function(key, value) {       
        OBP.updateHTMLFromXMLElement('setup', key, value, xml);
    });
  },
};

//-----------------------------
$(document).ready(OBP.init);
